/**********************************************************************
*	Bastion Products Copyright 1998								*
*	All rights reserved										*
*	This source is freely distributable in an unmodified state.		*
*	Project:			Blithe Shared Library						*
*	Source:			BlitheSetApp.cpp							*
*	Last Modified:	5/23/98									*
*	Author:			Jennifer Weston							*
*	Description:		This project creates a shared library which	*
*					allows applications to have dynamic text in	*
*					different languages.						*
***********************************************************************/
#include "BlitheSetApp.h"
#include <Path.h>
#include <FindDirectory.h>
#include <Resources.h>
#include <TextView.h>
#include <MenuField.h>
#include <CheckBox.h>

#include <cstdlib>

/**********************************************************************
*	Method:		BlitheSetApp (constructor)						*
*	Arguments:	<none>										*
*	Returns:		<none>										*
*	Description:	Creates the BlitheDictionary object and creates 	*
*				and sets up the settings window.					*
**********************************************************************/
BlitheSetApp::BlitheSetApp() : BApplication("application/x-Bastion.BlitheSettings")
{
	appDictionary = new BlitheDictionary();
}

/**********************************************************************
*	Method:		~BlitheSetApp (destructor)						*
*	Arguments:	<none>										*
*	Returns:		<none>										*
*	Description:	Deletes the application dictionary object			*
**********************************************************************/
BlitheSetApp::~BlitheSetApp()
{
	delete appDictionary;
}

/**********************************************************************
*	Method:		ReadyToRun									*
*	Arguments:	<none>										*
*	Returns:		<none>										*
*	Description:	Sets up the window								*
**********************************************************************/
void BlitheSetApp::ReadyToRun(void)
{
	settingsWindow = new BlitheSetWindow(appDictionary);
	BlitheUpdate(get_default_language());
	settingsWindow->Show();
}

/**********************************************************************
*	Method:		MessageReceived								*
*	Arguments:	BMessage*		message							*
*								The incoming message			*
*	Returns:		<none>										*
*	Description:	Overrides the BApplication::MessageReceived.		*
*				Handles BLITHE_LANGUAGE_CHANGED and				*
*				BLITHE_UPDATE_LANGUAGE message.					*
**********************************************************************/
void BlitheSetApp::MessageReceived(BMessage *message)
{
	status_t err;
	int32 id;

	switch(message->what)
	{
	case BLITHE_LANGUAGE_CHANGED:
	{
	// Get the id from the message
		err = message->FindInt32("langid",&id);
		if (err != B_OK) break;

	// If the check box is on, notify all running Blithe-aware apps
		bool notify = settingsWindow->ShouldNotify();

	// Get the position of the window
		BRect frame = settingsWindow->Frame();
		BPoint pos;
		pos.x = frame.left;
		pos.y = frame.top;

	// Set the id in the Blithe Settings file
		BPath settingsPath;
		err = find_directory(B_USER_SETTINGS_DIRECTORY,&settingsPath);
		if (err == B_NO_ERROR)
		{
			settingsPath.Append("blithe_settings");
			BFile settingsFile(settingsPath.Path(),B_READ_WRITE|B_CREATE_FILE|B_ERASE_FILE);
			BResources settingsRes(&settingsFile,true);
			err = settingsRes.AddResource('LONG',1,&id,sizeof(int32));
			err = settingsRes.AddResource('blnt',1,&notify,sizeof(bool));
			err = settingsRes.AddResource('winp',1,&pos,sizeof(pos));
		}

		// Broadcast message to all apps BLITHE_UPDATE LANGUAGE
		if (notify)
		{
			BMessage *myMessage = new BMessage(BLITHE_UPDATE_LANGUAGE);
			myMessage->AddInt32("langid",id);
			be_roster->Broadcast(myMessage);
		}
	}
		break;
	case BLITHE_UPDATE_LANGUAGE:
	// Update our fields
		err = message->FindInt32("langid",&id);
		if (err != B_OK) break;
		BlitheUpdate(id);
		break;
	default:
		BApplication::MessageReceived(message);
	}
}

/**********************************************************************
*	Method:		BlitheUpdate									*
*	Arguments:	int32	id									*
*							The new language ID				*
*	Returns:		<none>										*
*	Description:	Updates all text in the settings window			*
**********************************************************************/
void BlitheSetApp::BlitheUpdate(int32 id)
{
// Lock the window
	if (!settingsWindow) return;
	if (!settingsWindow->Lock()) return;

// Update the title of the app
	settingsWindow->SetTitle(appDictionary->Entry(BLITHE_SETTING_STRING,id));

// Update the instruction string
	BTextView *theTextView = dynamic_cast<BTextView*>(settingsWindow->FindView("instructions"));
	if (theTextView) theTextView->SetText(appDictionary->Entry(BLITHE_INSTR_STRING,id));

// Update the language label
	BMenuField *theMField = dynamic_cast<BMenuField*>(settingsWindow->FindView("menubar"));
	if (theMField)
	{
		char *theText;
		theMField->SetLabel(theText = appDictionary->Entry(BLITHE_LANGUAGE_STRING,id));
		float theWidth = theMField->StringWidth(theText) + 10;
		theMField->SetDivider(theWidth);
		float theHeight = theMField->Bounds().Height();
		theMField->ResizeTo(theWidth+10+theMField->MenuBar()->Bounds().Width(),theHeight);
	}

// Update the notify text
	BCheckBox *theCheckBox = dynamic_cast<BCheckBox*>(settingsWindow->FindView("notify"));
	if (theCheckBox) theCheckBox->SetLabel(appDictionary->Entry(BLITHE_NOTIFY_STRING,id));
	settingsWindow->Unlock();
}